--[[----------------------------------------------------------------------------
JudgeMyJPEG - Analyze Photo
Analyse la photo sélectionnée dans Lightroom
------------------------------------------------------------------------------]]

local LrApplication = import 'LrApplication'
local LrDialogs = import 'LrDialogs'
local LrTasks = import 'LrTasks'
local LrLogger = import 'LrLogger'
local LrFunctionContext = import 'LrFunctionContext'
local LrProgressScope = import 'LrProgressScope'

local JudgeMyJPEGAPI = require 'JudgeMyJPEGAPI'

local logger = LrLogger('JudgeMyJPEG')
logger:enable('logfile')

--[[----------------------------------------------------------------------------
Fonction principale d'analyse
------------------------------------------------------------------------------]]
LrTasks.startAsyncTask(function()
	LrFunctionContext.callWithContext('analyzePhoto', function(context)
		local catalog = LrApplication.activeCatalog()
		local targetPhoto = catalog:getTargetPhoto()

		if not targetPhoto then
			LrDialogs.message('Aucune photo sélectionnée', 'Veuillez sélectionner une photo à analyser.', 'info')
			return
		end

		-- Progress indicator
		local progressScope = LrProgressScope({
			title = 'JudgeMyJPEG - Analyse en cours',
			functionContext = context,
		})

		progressScope:setCaption('Préparation de la photo...')

		-- Récupérer l'URL de la photo
		-- IMPORTANT: Pour que ça fonctionne, la photo doit être accessible via une URL
		-- Option 1: Demander l'URL à l'utilisateur
		-- Option 2: Upload automatique vers Cloudinary (à implémenter)
		-- Option 3: Utiliser une photo déjà publiée

		progressScope:setCaption('Demande de l\'URL de la photo...')

		local photoUrl = LrDialogs.presentTextDialog({
			title = 'URL de la photo',
			message = 'Pour analyser cette photo, elle doit être accessible en ligne.\n\n' ..
				'Veuillez entrer l\'URL publique de la photo\n' ..
				'(ex: depuis Cloudinary, Imgur, ou votre site web)',
			defaultValue = '',
			actionVerb = 'Analyser',
		})

		if not photoUrl or photoUrl == '' then
			progressScope:done()
			return
		end

		progressScope:setCaption('Analyse de la photo avec l\'IA...')
		progressScope:setPortionComplete(0.3, 1)

		-- Appel API
		local result, errorMsg = JudgeMyJPEGAPI.analyzePhoto(photoUrl, nil)

		if errorMsg then
			progressScope:done()
			LrDialogs.message('Erreur', errorMsg, 'critical')
			return
		end

		progressScope:setPortionComplete(1, 1)
		progressScope:done()

		-- Affichage des résultats
		local score = result.result.score
		local summary = result.result.summary
		local recommendations = result.result.recommendations or {}
		local categories = result.result.categories or {}

		-- Construction du message de résultat
		local message = string.format('Score global: %d/100\n\n', score)

		-- Scores par catégorie
		if categories.composition then
			message = message .. string.format('📐 Composition: %d/20\n', categories.composition)
		end
		if categories.light then
			message = message .. string.format('💡 Lumière: %d/20\n', categories.light)
		end
		if categories.focus then
			message = message .. string.format('🎯 Netteté: %d/20\n', categories.focus)
		end
		if categories.color then
			message = message .. string.format('🎨 Couleurs: %d/20\n', categories.color)
		end
		if categories.creativity then
			message = message .. string.format('✨ Créativité: %d/20\n', categories.creativity)
		end

		message = message .. '\n' .. summary

		-- Recommandations
		if #recommendations > 0 then
			message = message .. '\n\n📝 Recommandations:\n'
			for i, rec in ipairs(recommendations) do
				message = message .. '• ' .. rec .. '\n'
			end
		end

		-- Affichage
		LrDialogs.message(
			'Analyse JudgeMyJPEG - ' .. score .. '/100',
			message,
			'info'
		)

		-- Logger le résultat
		logger:info(string.format('Analyse terminée - Score: %d/100', score))
	end)
end)
