--[[----------------------------------------------------------------------------
JudgeMyJPEG - API Client
Gère les appels à l'API MCP
------------------------------------------------------------------------------]]

local LrHttp = import 'LrHttp'
local LrPathUtils = import 'LrPathUtils'
local LrFileUtils = import 'LrFileUtils'
local LrDialogs = import 'LrDialogs'
local LrLogger = import 'LrLogger'
local json = require 'json'

local logger = LrLogger('JudgeMyJPEG')
logger:enable('logfile')

local JudgeMyJPEGAPI = {}

-- Configuration
JudgeMyJPEGAPI.MCP_BASE_URL = 'https://mcp-server-production-57ab.up.railway.app'

--[[----------------------------------------------------------------------------
Lit le token MCP depuis les préférences
------------------------------------------------------------------------------]]
function JudgeMyJPEGAPI.getToken()
	local prefs = import 'LrPrefs'.prefsForPlugin()
	return prefs.mcpToken
end

--[[----------------------------------------------------------------------------
Lit les paramètres d'analyse depuis les préférences
------------------------------------------------------------------------------]]
function JudgeMyJPEGAPI.getSettings()
	local prefs = import 'LrPrefs'.prefsForPlugin()
	return {
		tone = prefs.analysisMode or 'professional',
		language = prefs.language or 'fr',
	}
end

--[[----------------------------------------------------------------------------
Exporte temporairement une photo pour l'upload
@param photo: LrPhoto object
@return: chemin du fichier temporaire
------------------------------------------------------------------------------]]
function JudgeMyJPEGAPI.exportPhotoTemp(photo)
	local tempPath = LrPathUtils.getStandardFilePath('temp')
	local photoName = LrPathUtils.leafName(photo:getFormattedMetadata('fileName'))
	local exportPath = LrPathUtils.child(tempPath, 'JudgeMyJPEG_' .. photoName)

	-- Export en JPEG qualité 90%
	local exportSettings = {
		LR_export_destinationType = 'specificFolder',
		LR_export_destinationPathPrefix = tempPath,
		LR_export_useSubfolder = false,
		LR_format = 'JPEG',
		LR_jpeg_quality = 0.9,
		LR_size_maxWidth = 2048,
		LR_size_maxHeight = 2048,
		LR_size_resizeType = 'wh',
	}

	-- TODO: Implémenter export réel avec LrExportSession
	-- Pour l'instant, on retourne le chemin du fichier original
	return photo:getRawMetadata('path')
end

--[[----------------------------------------------------------------------------
Upload une photo vers Cloudinary et retourne l'URL
@param filePath: chemin du fichier
@return: URL Cloudinary ou nil
------------------------------------------------------------------------------]]
function JudgeMyJPEGAPI.uploadToCloudinary(filePath)
	-- Pour simplifier, on va d'abord demander l'URL à l'utilisateur
	-- Dans une version future, on pourrait uploader vers Cloudinary directement
	logger:info('Upload vers Cloudinary: ' .. filePath)

	-- TODO: Implémenter upload Cloudinary
	return nil
end

--[[----------------------------------------------------------------------------
Analyse une photo via l'API MCP
@param photoUrl: URL de la photo (ou nil pour base64)
@param photoPath: chemin local (pour base64)
@return: résultat de l'analyse ou nil + message d'erreur
------------------------------------------------------------------------------]]
function JudgeMyJPEGAPI.analyzePhoto(photoUrl, photoPath)
	local token = JudgeMyJPEGAPI.getToken()

	if not token or token == '' then
		return nil, 'Token MCP non configuré. Allez dans Fichier > Gestionnaire de modules externes > JudgeMyJPEG'
	end

	local settings = JudgeMyJPEGAPI.getSettings()

	-- Construction du body
	local requestBody = {
		photo_url = photoUrl,
		options = settings,
	}

	-- Si pas d'URL, il faudrait encoder en base64
	-- Pour l'instant on requiert une URL
	if not photoUrl then
		return nil, 'Veuillez d\'abord uploader la photo sur un serveur accessible (ex: Cloudinary, Imgur)'
	end

	local jsonBody = json.encode(requestBody)

	-- Headers
	local headers = {
		{ field = 'Content-Type', value = 'application/json' },
		{ field = 'x-mcp-token', value = token },
	}

	logger:info('Appel MCP /analyze: ' .. photoUrl)

	-- Requête HTTP POST
	local response, hdrs = LrHttp.post(
		JudgeMyJPEGAPI.MCP_BASE_URL .. '/mcp/analyze',
		jsonBody,
		headers,
		'POST',
		30 -- timeout 30s
	)

	-- Vérification du statut
	local status = nil
	for _, h in ipairs(hdrs) do
		if h.field == 'status' then
			status = tonumber(h.value)
			break
		end
	end

	logger:info('Statut réponse: ' .. tostring(status))

	if status ~= 200 then
		logger:error('Erreur API: ' .. tostring(response))
		return nil, 'Erreur API (' .. tostring(status) .. '): ' .. tostring(response)
	end

	-- Parse JSON
	local success, result = pcall(json.decode, response)

	if not success then
		logger:error('Erreur parse JSON: ' .. tostring(result))
		return nil, 'Erreur lors du parsing de la réponse'
	end

	logger:info('Analyse réussie: score = ' .. tostring(result.result.score))
	return result, nil
end

--[[----------------------------------------------------------------------------
Compare deux photos via l'API MCP
@param photoUrlA: URL de la photo A
@param photoUrlB: URL de la photo B
@return: résultat de la comparaison ou nil + message d'erreur
------------------------------------------------------------------------------]]
function JudgeMyJPEGAPI.comparePhotos(photoUrlA, photoUrlB)
	local token = JudgeMyJPEGAPI.getToken()

	if not token or token == '' then
		return nil, 'Token MCP non configuré'
	end

	local requestBody = {
		photo_a = photoUrlA,
		photo_b = photoUrlB,
	}

	local jsonBody = json.encode(requestBody)

	local headers = {
		{ field = 'Content-Type', value = 'application/json' },
		{ field = 'x-mcp-token', value = token },
	}

	logger:info('Appel MCP /compare')

	local response, hdrs = LrHttp.post(
		JudgeMyJPEGAPI.MCP_BASE_URL .. '/mcp/compare',
		jsonBody,
		headers,
		'POST',
		30
	)

	local status = nil
	for _, h in ipairs(hdrs) do
		if h.field == 'status' then
			status = tonumber(h.value)
			break
		end
	end

	if status ~= 200 then
		logger:error('Erreur API compare: ' .. tostring(response))
		return nil, 'Erreur API (' .. tostring(status) .. ')'
	end

	local success, result = pcall(json.decode, response)

	if not success then
		return nil, 'Erreur parsing JSON'
	end

	return result, nil
end

return JudgeMyJPEGAPI
